﻿using log4net;
using Microsoft.Azure.KeyVault;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.IdentityModel.Clients.ActiveDirectory;
using System;
using System.IO;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Runtime.Serialization.Json;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.IWS.Api.AddressValidationService.Interface;
using VA.PPMS.IWS.Api.Configuration.Interface;
using VA.PPMS.IWS.Common;

namespace VA.PPMS.IWS.Api.AddressValidationService
{
    public class AddressValidationService : IAddressValidationService
    {
        private const string Entry = "_Secret";

        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IMemoryCache _cache;

        public AddressValidationService(ILog logger, IIwsConfiguration configuration, IMemoryCache cache)
        {
            _logger = logger;
            _configuration = configuration;
            _cache = cache;
        }

        public async Task<RootObjectResponse> ValidateAddress(RootObjectRequest request)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            _logger.Info("@@@@ INFO - Start Address Validation @@@@");

            try
            {
                using (var client = await GetHttpClient())
                {
                    client.BaseAddress = new Uri(await _configuration.GetAddressValidationBaseUri());

                    var json = Serialize(request);
                    _logger.Info("Address Validation Request");

                    var content = new StringContent(json, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeHeaderValue("application/json");

                    var response = await client.PostAsync(await _configuration.GetAddressValidationRequestUri(), content);

                    if (response.IsSuccessStatusCode)
                    {
                        var result = response.Content.ReadAsStringAsync().GetAwaiter().GetResult();
                        _logger.Info("Address Validation Response received");

                        if (!string.IsNullOrEmpty(result)) return Deserialize<RootObjectResponse>(result);

                        _logger.Error("@@@@ ERROR - The results of the Address Validation call are null");
                        return default(RootObjectResponse);
                    }

                    _logger.Error("@@@@ ERROR - The Response from Address Validation Service does not indicate success");
                    return default(RootObjectResponse);
                }
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the Address Validation Service ", ex);
                throw new Exception("There was a problem with the Address Validation Service", ex);
            }
        }

        private async Task<HttpClient> GetHttpClient()
        {
            var clientHandler = new HttpClientHandler();
            clientHandler.ClientCertificates.Add(await GetCertKeyVault());

            return new HttpClient(clientHandler);
        }

        private async Task<X509Certificate2> GetCertKeyVault()
        {
            if (_cache.TryGetValue(Entry, out X509Certificate2 cacheEntry))
            {
                _logger.Info("Found certificate in cache");
                return cacheEntry;
            }

            var keyVaultClient = new KeyVaultClient(GetToken);
            var result = await keyVaultClient.GetSecretAsync(await _configuration.GetAddressValidationCertUrl());

            if (result == null) throw new InvalidOperationException("Failed to obtain the certificate from Key Vault");

            var secret = Convert.FromBase64String(result.Value);
            cacheEntry = new X509Certificate2(secret, (string)null);

            var cacheEntryOptions = new MemoryCacheEntryOptions().SetSlidingExpiration(TimeSpan.FromHours(1));
            _cache.Set(Entry, cacheEntry, cacheEntryOptions);

            return cacheEntry;
        }

        private async Task<string> GetToken(string authority, string resource, string scope)
        {
            var authContext = new AuthenticationContext(authority);
            var clientCred = new ClientCredential(await _configuration.GetAddressValidationAppId(), await _configuration.GetAddressValidationSecret());
            var result = await authContext.AcquireTokenAsync(resource, clientCred);

            if (result == null) throw new InvalidOperationException("Failed to obtain the token to retrieve certificate");

            return result.AccessToken;
        }

        private static string Serialize<T>(T data)
        {
            var ms = new MemoryStream();
            var ser = new DataContractJsonSerializer(typeof(T));
            ser.WriteObject(ms, data);
            var json = ms.ToArray();
            ms.Close();

            return Encoding.UTF8.GetString(json, 0, json.Length);
        }

        private static T Deserialize<T>(string json)
        {
            var ms = new MemoryStream(Encoding.UTF8.GetBytes(json));
            var ser = new DataContractJsonSerializer(typeof(T));
            var result = (T)ser.ReadObject(ms);
            ms.Close();

            return result;
        }
    }
}